﻿using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using System.Web;
using System.Web.Http;
using System.Web.OData;
using System.Web.OData.Routing;
using Microsoft.Xrm.Sdk;
using PpmsDataService.V1.Mappers;
using PpmsDataService.Models;
using PpmsDataService.VA.PPMS.Context;
using VA.PPMS.Context;
using System.Net;
using System.Net.Http;
using System.Web.Http.Description;
using Microsoft.Web.Http;

namespace PpmsDataService.V1.Controllers
{
    [EnableQuery]
    //[Authorize]
    [ApiVersion("1.0")]
    public class FacilitiesController : ODataController
    {
        [ODataRoute("Facilities('{key}')")]
        public async Task<HttpResponseMessage> Get([FromODataUri] string key)
        {
            //We will use the Query type found in URL to determine if Expanded properties need to be mapped. 
            var queryType = HttpContext.Current.Request.Url.Query;

            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //Retrieve the Facility 
                var ppmsFacility = context.va_facilitySet.FirstOrDefault(i => i.va_stationnumber.Equals(key));
                if (ppmsFacility != null)
                {
                    var ppmsFacilitiesList = new List<va_facility> { ppmsFacility };
                    var facilities = await FacilityMap.MapFacilities(ppmsFacilitiesList, queryType, context);
                    return Request.CreateResponse(facilities);
                }
            }
            var message = string.Format("Facility with Station Number: {0} not found", key);
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }

        [ODataRoute("Facilities('{key}')/Visn")]
        public async Task<HttpResponseMessage> GetVisn([FromODataUri] string key)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //Retrieve the Facility 
                var ppmsFacility = context.va_facilitySet.FirstOrDefault(i => i.va_stationnumber.Equals(key));
                if (ppmsFacility != null)
                {
                    //Get the Related Visn
                    var ppmsVisn =
                        context.va_visnSet.FirstOrDefault(i => i.Id == ppmsFacility.va_visnid.Id);
                    var ppmsVisnList = new List<va_visn> { ppmsVisn };
                    var visn =
                        await VisnMap.MapVisns(ppmsVisnList, "None", context);
                    return Request.CreateResponse(visn);
                }
            }
            var message = string.Format("Visn with Facility Station Number: {0} not found", key);
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }

        [ODataRoute("Facilities('{key}')/ParentStation")]
        public async Task<HttpResponseMessage> GetParentStation([FromODataUri] string key)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //Retrieve the Facility 
                var ppmsFacility = context.va_facilitySet.FirstOrDefault(i => i.va_stationnumber.Equals(key));
                if (ppmsFacility != null)
                {
                    //Get the Related Parent Station
                    var ppmsParentStation =
                        context.va_facilitySet.FirstOrDefault(i => i.Id == ppmsFacility.ppms_ParentStation.Id);
                    var ppmsParentStationList = new List<va_facility> { ppmsParentStation };
                    var parentStation =
                        await FacilityMap.MapFacilities(ppmsParentStationList, "None", context);
                    return Request.CreateResponse(parentStation);
                }
            }
            var message = string.Format("Parent Station with Facility Station Number: {0} not found", key);
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }

        [ODataRoute("Facilities('{key}')/Facilities")]
        public async Task<HttpResponseMessage> GetFacilities([FromODataUri] string key)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //Retrieve the Facility 
                var ppmsFacility = context.va_facilitySet.FirstOrDefault(i => i.va_stationnumber.Equals(key));
                if (ppmsFacility != null)
                {
                    //Get the Related Facilities
                    var ppmsRelatedFacilities =
                        context.va_facilitySet.Where(i => i.ppms_ParentStation.Id == ppmsFacility.Id);
                    var ppmsRelatedFacilitiesList = ppmsRelatedFacilities.ToList();
                    var facilities =
                        await FacilityMap.MapFacilities(ppmsRelatedFacilitiesList, "None", context);
                    return Request.CreateResponse(facilities);
                }
            }
            var message = string.Format("Related Facilities with Parent Facility Station Number: {0} not found", key);
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }

        [ODataRoute("Facilities('{key}')/CareSites")]
        public async Task<HttpResponseMessage> GetCareSites([FromODataUri] string key)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //Retrieve the Facility 
                var ppmsFacility = context.va_facilitySet.FirstOrDefault(i => i.va_stationnumber.Equals(key));
                if (ppmsFacility != null)
                {
                    //Get the Related Care Sites
                    var ppmsCareSites =
                        context.ppms_caresiteSet.Where(i => i.ppms_facilityid.Id == ppmsFacility.Id);
                    var ppmsCareSitesList = ppmsCareSites.ToList();
                    var careSites =
                        await CareSiteMap.MapCareSites(ppmsCareSitesList, "None", context);
                    return Request.CreateResponse(careSites);
                }
            }
            var message = string.Format("Care Site with Facilty Station Number: {0} not found", key);
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }

        [ODataRoute("Facilities('{key}')/Providers")]
        public async Task<HttpResponseMessage> GetProviders([FromODataUri] string key)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //Retrieve the Facility 
                var ppmsFacility = context.va_facilitySet.FirstOrDefault(i => i.va_stationnumber.Equals(key));
                if (ppmsFacility != null)
                {
                    //Get the Related Providers
                    var ppmsProviders =
                        context.AccountSet.Where(i => i.ppms_facilityid.Id == ppmsFacility.Id);
                    var ppmsProvidersList = ppmsProviders.ToList();
                    var providers =
                        await ProviderMap.MapProviders(ppmsProvidersList, "None", context);
                    return Request.CreateResponse(providers);
                }
            }
            var message = string.Format("Providers with Facility Station Number: {0} not found", key);
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }

        [HttpGet]
        [MapToApiVersion("1.0")]
        [ResponseType(typeof(Facility))]
        [ODataRoute("GetFacilityByCity")]
        public async Task<HttpResponseMessage> GetFacilityByCity([FromODataUri] string city)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                var ppmsFacilities = context.va_facilitySet.Where(i => i.va_city.Contains(city));
                var ppmsFacilitiesList = ppmsFacilities.ToList();
                if (ppmsFacilitiesList.Any())
                {
                    //Map Facilities
                    var facilities = await FacilityMap.MapFacilities(ppmsFacilitiesList, "None", context);
                    return Request.CreateResponse(facilities);
                }
                var message = string.Format("Facilities with City: {0} not found", city);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [HttpGet]
        [MapToApiVersion("1.0")]
        [ResponseType(typeof(Facility))]
        [ODataRoute("GetFacilityByState")]
        public async Task<HttpResponseMessage> GetFacilityByState([FromODataUri] string state)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                var ppmsFacilities = context.va_facilitySet.Where(i => i.va_state.Contains(state));
                var ppmsFacilitiesList = ppmsFacilities.ToList();
                if (ppmsFacilitiesList.Any())
                {
                    //Map Facilities
                    var facilities = await FacilityMap.MapFacilities(ppmsFacilitiesList, "None", context);
                    return Request.CreateResponse(facilities);
                }
                var message = string.Format("Facilities with State: {0} not found", state);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [HttpGet]
        [MapToApiVersion("1.0")]
        [ResponseType(typeof(Facility))]
        [ODataRoute("GetFacilityByZipCode")]
        public async Task<HttpResponseMessage> GetFacilityByZipCode([FromODataUri] string zip)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                var ppmsFacilities = context.va_facilitySet.Where(i => i.va_zipcode.Contains(zip));
                var ppmsFacilitiesList = ppmsFacilities.ToList();
                if (ppmsFacilitiesList.Any())
                {
                    //Map Facilities
                    var facilities = await FacilityMap.MapFacilities(ppmsFacilitiesList, "None", context);
                    return Request.CreateResponse(facilities);
                }
                var message = string.Format("Facilities with Zip Code: {0} not found", zip);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [HttpGet]
        [MapToApiVersion("1.0")]
        [ResponseType(typeof(Facility))]
        [ODataRoute("GetFacilityByRegion")]
        public async Task<HttpResponseMessage> GetFacilityByRegion([FromODataUri] string region)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                var ppmsFacilities = context.va_facilitySet.Where(i => i.ppms_regionname.Contains(region));
                var ppmsFacilitiesList = ppmsFacilities.ToList();
                if (ppmsFacilitiesList.Any())
                {
                    //Map Facilities
                    var facilities = await FacilityMap.MapFacilities(ppmsFacilitiesList, "None", context);
                    return Request.CreateResponse(facilities);
                }
                var message = string.Format("Facilities with Region Name: {0} not found", region);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("Facilities")]
        public async Task<HttpResponseMessage> Get()
        {
            //This Scenario returns the first 50 Providers in the system. . 
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                var ppmsFacilities = context.va_facilitySet.Where(i => i.va_stationnumber != null).Take(50);
                var ppmsFacilitiesList = ppmsFacilities.ToList();
                if (ppmsFacilitiesList.Any())
                {
                    //Map facilities
                    var facilities = await FacilityMap.MapFacilities(ppmsFacilitiesList, "None", context);
                    return Request.CreateResponse(facilities);
                }
                var message = "Facilities not found";
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }
    }
}